/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2005 by Myricom, Inc.  All rights reserved.                 *
 *************************************************************************/

#include <unistd.h>
#include <sys/time.h>
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include "mx_auto_config.h"
#include "myriexpress.h"
#include "mx_byteswap.h"

#define MXFUNC(mx_func, abort_loc) \
do { if ((mx_func) != MX_SUCCESS) goto abort_loc; } while (0)

/* Flood mode */
#define FLOOD_MASTER 1
#define FLOOD_SLAVE 2
/* Flood direction */
#define FLOOD_UNIDIRECTIONAL 3
#define FLOOD_BIDIRECTIONAL 4

#define FLOOD_EID 2
#define FLOOD_KEY 0x6b6579
#define FLOOD_PARAM UINT64_C(0x706172616d)
#define FLOOD_PARAM_REPLY UINT64_C(0x7265706c79)

struct flood_options
{
  int duty; /* Master or slave */
  int direction; /* Uni/bidirectional */
  char *hostname; /* Hostname of slave, NULL for slave */
  uint32_t board;
  uint32_t my_eid;
  uint32_t his_eid;
  uint32_t key;
  uint32_t len;
  uint32_t iter;
  uint32_t verify;
  uint32_t verbose;
  uint32_t wait;
};

struct flood_params
{
  uint16_t high16; /* high 16 bits of master nic id */
  uint16_t pad1;
  uint32_t low32; /* low 32 bits of master nic id */
  uint32_t eid; /* eid that slave should use to connect to the master */
  uint32_t direction;
  uint32_t len;
  uint32_t iter;
  uint32_t verify;
  uint32_t verbose;
  uint32_t wait;
};

static void usage(void)
{
  fprintf(stderr, "Usage: mx_flood [args]\n");
  fprintf(stderr, "-d hostname - destination hostname, required for sender\n");
  fprintf(stderr, "-x          - bidirectional\n");
}

#define FLOOD_DFLT_LEN 40000
#define FLOOD_DFLT_ITER 1000

static void init_default_options(struct flood_options *opts)
{
  opts->duty = FLOOD_SLAVE;
  opts->direction = FLOOD_UNIDIRECTIONAL;
  opts->hostname = NULL;
  opts->board = 0;
  opts->my_eid = FLOOD_EID;
  opts->his_eid = FLOOD_EID;
  opts->key = FLOOD_KEY;
  opts->len = FLOOD_DFLT_LEN;
  opts->iter = FLOOD_DFLT_ITER;
  /* TODO: verify */
  /* TODO: verbose */
  /* TODO: wait */
  opts->verify = 0;
  opts->verbose = 0;
  opts->wait = 0;
}

static int parse_args(int argc, char **argv, struct flood_options *opts)
{
  int opt;

  uint64_t nic_id;

  while ((opt = getopt(argc, argv, "hd:e:n:b:r:l:f:N:Vvwx")) != -1) {
    switch (opt) {
    case 'd':
      opts->duty = FLOOD_MASTER;
      opts->hostname = optarg;
      break;
    case 'e':
      opts->my_eid = atoi(optarg);
      break;
    case 'x':
      opts->direction = FLOOD_BIDIRECTIONAL;
      break;
    case 'n':
      nic_id = strtoull(optarg, NULL, 16);
      MXFUNC((mx_nic_id_to_board_number(nic_id, &opts->board)),
	      abort_with_nothing);
      break;
    case 'b':
      opts->board = atoi(optarg);
      break;
    case 'r':
      opts->his_eid = atoi(optarg);
      break;
    case 'l':
      opts->len = atoi(optarg);
      break;
    case 'N':
      opts->iter = atoi(optarg);
      break;
    case 'f':
      opts->key = atoi(optarg);
      break;
    case 'V':
      opts->verify = 1;
      break;
    case 'v':
      opts->verbose = 1;
      break;
    case 'w':
      opts->wait = 1;
      break;
    case 'h':
      opts->his_eid = atoi(optarg);
      break;
    default:
      usage();
      goto abort_with_nothing;
    }
  }

  return 0;

 abort_with_nothing:
  return 1;
}

static int receive_opts(mx_endpoint_t ep, struct flood_options *opts,
			mx_endpoint_addr_t *dest_addr)
{
  struct flood_params params;
  uint64_t nic_id;
  uint32_t eid;
  mx_segment_t seg;
  mx_request_t req;
  mx_status_t status;
  uint32_t result;

  seg.segment_ptr = &params;
  seg.segment_length = sizeof (params);

  MXFUNC((mx_irecv(ep, &seg, 1, FLOOD_PARAM, MX_MATCH_MASK_NONE, NULL,
		    &req)),
	  abort_with_nothing);
  MXFUNC((mx_wait(ep, &req, MX_INFINITE, &status, &result)),
	  abort_with_nothing);

  opts->direction = ntohl(params.direction);
  opts->len = ntohl(params.len);
  opts->iter = ntohl(params.iter);
  opts->verify = ntohl(params.verify);
  opts->verbose = ntohl(params.verbose);
  opts->wait = ntohl(params.wait);

  nic_id = ntohs(params.high16);
  nic_id <<= 32;
  nic_id |= ntohl(params.low32);
  eid = ntohl(params.eid);
  MXFUNC((mx_connect(ep, nic_id, eid, opts->key, MX_INFINITE,
		      dest_addr)),
	  abort_with_nothing);

  return 0;

 abort_with_nothing:
  return 1;
}

static int send_opts(mx_endpoint_t ep, struct flood_options *opts,
		     mx_endpoint_addr_t *dest_addr)
{
  struct flood_params params;
  mx_endpoint_addr_t addr;
  uint64_t dest_nic_id;
  uint64_t nic_id;
  uint32_t eid;
  mx_segment_t seg;
  mx_request_t req;
  mx_status_t status;
  uint32_t result;

  MXFUNC((mx_hostname_to_nic_id(opts->hostname, &dest_nic_id)),
	  abort_with_nothing);

  MXFUNC((mx_connect(ep, dest_nic_id, opts->his_eid, opts->key,
		      MX_INFINITE, dest_addr)),
	  abort_with_nothing);

  MXFUNC((mx_get_endpoint_addr(ep, &addr)), abort_with_nothing);
  MXFUNC((mx_decompose_endpoint_addr(addr, &nic_id, &eid)),
	  abort_with_nothing);

  params.high16 = htons((uint16_t)(nic_id >> 32));
  params.low32 = htonl((uint32_t)nic_id);
  params.eid = htonl(eid);
  params.direction = htonl(opts->direction);
  params.len = htonl(opts->len);
  params.iter = htonl(opts->iter);
  params.verify = htonl(opts->verify);
  params.verbose = htonl(opts->verbose);
  params.wait = htonl(opts->wait);

  seg.segment_ptr = &params;
  seg.segment_length = sizeof (params);

  MXFUNC((mx_issend(ep, &seg, 1, *dest_addr, FLOOD_PARAM, NULL, &req)),
	  abort_with_nothing);
  MXFUNC((mx_wait(ep, &req, MX_INFINITE, &status, &result)),
	  abort_with_nothing);

  /* The slave is supposed to connect back. Give it some time. */
  sleep(1);

  return 0;

 abort_with_nothing:
  return 1;
}

#define FLOOD_SEND_COUNT 16
#define FLOOD_RECV_COUNT 16
#define FLOOD_MAGIC 0x42
#define FLOOD_GO 0x43
#define FLOOD_USEC 0x44

#define FLOOD_SEND 0x10000000

#define IPEEK_OR_PEEK(opts_, ep_, req_, result_, x_)                   \
do {                                                                   \
  if(opts_->wait == 1) {                                               \
    MXFUNC((mx_peek(ep_, MX_INFINITE, req_, result_)),                \
	    abort_with_nothing);                                       \
  }                                                                    \
  else {                                                               \
    do {                                                               \
      MXFUNC((mx_ipeek(ep_, req_, result_)), abort_with_nothing);     \
    } while (result == 0);                                             \
  }                                                                    \
  MXFUNC((mx_context(req_, x_)), abort_with_nothing);                 \
} while (0)

static void fill_buffer(mx_segment_t *seg, uint32_t val)
{
  uint32_t i;
  uint32_t *p;

  p = (uint32_t*)seg->segment_ptr;
  for (i = 0; i < (seg->segment_length/sizeof (val)); ++i) {
    *p++ = val;
  }
}

static void check_buffer(mx_segment_t *seg)
{
  uint32_t i;
  uint32_t val;
  uint32_t *p;

  if (seg->segment_length > sizeof (uint32_t)) {
    p = (uint32_t*)seg->segment_ptr;
    val = *p;
    for (i = 0; i < (seg->segment_length/sizeof (val)); ++i) {
      if (*p != val) {
	fprintf(stderr, "verify failed, expected %d, got %d at %d\n",
		(int)ntohl(val), (int)ntohl(*p), (int)(i * sizeof(val)));
	abort();
      }
    }
  }
}

static int flood(struct flood_options *opts)
{
  int ret = 0;
  mx_endpoint_t ep;
  mx_endpoint_addr_t dest_addr;

  uint32_t i;
  mx_segment_t *send_seg;
  mx_segment_t *recv_seg;
  char **send_buff;
  char **recv_buff;
  mx_request_t *send_req;
  mx_request_t *recv_req;

  mx_segment_t send_meta_seg;
  mx_segment_t recv_meta_seg;
  mx_request_t send_meta_req;
  mx_request_t recv_meta_req;

  mx_status_t status;
  uint32_t result;

  mx_request_t peek_req;
  void *x;

  struct timeval start_time;
  struct timeval end_time;
  uint32_t usec;
  double bw;
  double pkts_per_sec;

  int iter = 0;

  int send_posted = 0;
  int recv_posted = 0;
  int send_completed = 0;
  int recv_completed = 0;

  MXFUNC((mx_init()), abort_with_nothing);
  MXFUNC((mx_open_endpoint(opts->board, opts->my_eid, opts->key, NULL,
			    0, &ep)), abort_with_nothing);

  switch(opts->duty) {
  case FLOOD_SLAVE:
    ret = receive_opts(ep, opts, &dest_addr);
    break;
  case FLOOD_MASTER:
    ret = send_opts(ep, opts, &dest_addr);
    break;
  default:
    assert(0);
    ret = 1;
    break;
  }

  send_seg = malloc(sizeof (*send_seg) * FLOOD_SEND_COUNT);
  recv_seg = malloc(sizeof (*recv_seg) * FLOOD_RECV_COUNT);
  send_buff = malloc(sizeof (*send_buff) * FLOOD_SEND_COUNT);
  recv_buff = malloc(sizeof (*recv_buff) * FLOOD_RECV_COUNT);
  for (i = 0; i < FLOOD_SEND_COUNT; ++i) {
    send_buff[i] = malloc(opts->len);
    send_seg[i].segment_ptr = send_buff[i];
    send_seg[i].segment_length = opts->len;
  }
  for (i = 0; i < FLOOD_RECV_COUNT; ++i) {
    recv_buff[i] = malloc(opts->len);
    recv_seg[i].segment_ptr = recv_buff[i];
    recv_seg[i].segment_length = opts->len;
  }
  send_req = malloc(sizeof (*send_req) * FLOOD_SEND_COUNT);
  recv_req = malloc(sizeof (*recv_req) * FLOOD_RECV_COUNT);

  if (opts->duty == FLOOD_MASTER ||
      opts->direction == FLOOD_BIDIRECTIONAL) {
    iter += opts->iter;
  }
  if (opts->duty == FLOOD_SLAVE ||
      opts->direction == FLOOD_BIDIRECTIONAL) {
    iter += opts->iter;
  }

  if (opts->duty == FLOOD_SLAVE ||
      opts->direction == FLOOD_BIDIRECTIONAL) {
    /* Prepost receives. */
    for (i = 0; i < FLOOD_RECV_COUNT; ++i) {
      MXFUNC((mx_irecv(ep, &recv_seg[i], 1, FLOOD_MAGIC,
			MX_MATCH_MASK_NONE, (void*)i, &recv_req[i])),
	      abort_with_nothing);
      recv_posted++;
    }
  }

  if (opts->duty == FLOOD_MASTER ||
      opts->direction == FLOOD_BIDIRECTIONAL) {
    recv_meta_seg.segment_ptr = NULL;
    recv_meta_seg.segment_length = 0;
    MXFUNC((mx_irecv(ep, &recv_meta_seg, 1, FLOOD_GO,
		      MX_MATCH_MASK_NONE, NULL, &recv_meta_req)),
	    abort_with_nothing);
  }

  if (opts->duty == FLOOD_SLAVE ||
      opts->direction == FLOOD_BIDIRECTIONAL) {
    /* Send ok to flood message. */
    send_meta_seg.segment_ptr = NULL;
    send_meta_seg.segment_length = 0;
    MXFUNC((mx_issend(ep, &send_meta_seg, 1, dest_addr, FLOOD_GO, NULL,
		      &send_meta_req)),
	    abort_with_nothing);
    MXFUNC((mx_wait(ep, &send_meta_req, MX_INFINITE, &status,
		     &result)),
	    abort_with_nothing);
  }

  if (opts->duty == FLOOD_MASTER ||
      opts->direction == FLOOD_BIDIRECTIONAL) {
    MXFUNC((mx_wait(ep, &recv_meta_req, MX_INFINITE, &status,
		     &result)),
	    abort_with_nothing);
  }

  gettimeofday(&start_time, NULL);
  
  if (opts->duty == FLOOD_MASTER ||
      opts->direction == FLOOD_BIDIRECTIONAL) {
    /* Post sends. */
    for (i = 0; i < FLOOD_SEND_COUNT; ++i) {
      if (opts->verify) {
	fill_buffer(&send_seg[i], htonl(send_posted));
      }
      MXFUNC((mx_isend(ep, &send_seg[i], 1, dest_addr, FLOOD_MAGIC,
			(void*)(i | FLOOD_SEND), &send_req[i])),
	      abort_with_nothing);
      send_posted++;
    }
  }

  /******************************************************************/
  while ((send_posted + recv_posted) < iter) {
    IPEEK_OR_PEEK(opts, ep, &peek_req, &result, &x);
    if ((uintptr_t)x & FLOOD_SEND) {
      MXFUNC((mx_test(ep, &peek_req, &status, &result)),
	      abort_with_nothing);
      send_completed++;
      if (send_posted < opts->iter) {
	if (opts->verify) {
	  fill_buffer(&send_seg[(uintptr_t)x & ~FLOOD_SEND],
		      htonl(send_posted));
	}
	MXFUNC((mx_isend(ep, &send_seg[(uintptr_t)x & ~FLOOD_SEND],
			  1, dest_addr, FLOOD_MAGIC, x,
			  &send_req[(uintptr_t)x & ~FLOOD_SEND])),
		abort_with_nothing);
	send_posted++;
      }
    }
    else {
      if (opts->verify) {
	check_buffer(&recv_seg[(uintptr_t)x]);
      }
      MXFUNC((mx_test(ep, &peek_req, &status, &result)),
	      abort_with_nothing);
      recv_completed++;
      /* recv */
      if (recv_posted < opts->iter) {
	MXFUNC((mx_irecv(ep, &recv_seg[(uintptr_t)x], 1, FLOOD_MAGIC,
			  MX_MATCH_MASK_NONE, x,
			  &recv_req[(uintptr_t)x])),
		abort_with_nothing);
	recv_posted++;
      }
    }
  }

  while (send_completed + recv_completed < iter) {
    IPEEK_OR_PEEK(opts, ep, &peek_req, &result, &x);
    if ((uintptr_t)x & FLOOD_SEND) {
      MXFUNC((mx_test(ep, &peek_req, &status, &result)),
	      abort_with_nothing);
      send_completed++;
    }
    else {
      if (opts->verify) {
	check_buffer(&recv_seg[(uintptr_t)x]);
      }
      MXFUNC((mx_test(ep, &peek_req, &status, &result)),
	      abort_with_nothing);
      recv_completed++;
    }
  }

  gettimeofday(&end_time, NULL);

  usec = (uint32_t)(((uint64_t)end_time.tv_sec * 1000000 +
		     end_time.tv_usec) -
		    ((uint64_t)start_time.tv_sec * 1000000 +
		     start_time.tv_usec));
  if (opts->duty == FLOOD_SLAVE) {
    usec = htonl(usec);
    send_meta_seg.segment_ptr = &usec;
    send_meta_seg.segment_length = sizeof (usec);
    MXFUNC((mx_issend(ep, &send_meta_seg, 1, dest_addr, FLOOD_USEC,
		       NULL, &send_meta_req)),
	    abort_with_nothing);
    MXFUNC((mx_wait(ep, &send_meta_req, MX_INFINITE, &status,
		     &result)),
	    abort_with_nothing);
  }
  else {
    recv_meta_seg.segment_ptr = &usec;
    recv_meta_seg.segment_length = sizeof (usec);
    MXFUNC((mx_irecv(ep, &recv_meta_seg, 1, FLOOD_USEC,
		      MX_MATCH_MASK_NONE, NULL, &recv_meta_req)),
	    abort_with_nothing);
    MXFUNC((mx_wait(ep, &recv_meta_req, MX_INFINITE, &status,
		     &result)),
	    abort_with_nothing);
    usec = ntohl(usec);
    bw = (double)opts->iter * opts->len / usec;
    pkts_per_sec = (double)opts->iter / usec * 1000000.0;
    if (opts->direction == FLOOD_BIDIRECTIONAL) {
      bw *= 2;
      pkts_per_sec *= 2;
    }
    printf("%d %5.3f %5.3f\n", (int)usec, bw, pkts_per_sec);

    usec = (uint32_t)(((uint64_t)end_time.tv_sec * 1000000 +
		       end_time.tv_usec) -
		      ((uint64_t)start_time.tv_sec * 1000000 +
		       start_time.tv_usec));
    bw = (double)opts->iter * opts->len / usec;
    pkts_per_sec = (double)opts->iter / usec * 1000000.0;
    if (opts->direction == FLOOD_BIDIRECTIONAL) {
      bw *= 2;
      pkts_per_sec *= 2;
    }
    printf("%d %5.3f %5.3f\n", (int)usec, bw, pkts_per_sec);
  }

  MXFUNC((mx_finalize()), abort_with_nothing);

  return ret;

 abort_with_nothing:
  return ret;
}

int main(int argc, char **argv)
{
  int ret = 0;
  struct flood_options opts;

  init_default_options(&opts);
  if (parse_args(argc, argv, &opts) != 0) {
    ret = EXIT_FAILURE;
  }
  flood(&opts);

  return ret;
}
